<?php
/**
 * leopardAusgezeichnetOrg
 * Copyright (c) Die Leoparden e.K.
 */

namespace leopardAusgezeichnetOrg\Subscriber;

use Doctrine\DBAL\Query\QueryBuilder;
use Enlight\Event\SubscriberInterface;
use Enlight_Event_EventArgs;
use Enlight_Hook_HookArgs;
use Enlight_View_Default;
use Exception;
use PDO;
use Shopware\Components\Logger;
use Shopware\Models\Customer\Customer;
use Shopware\Models\Mail\Mail;
use Shopware\Models\Order\Order;
use Shopware\Models\Shop\Shop;

class Frontend implements SubscriberInterface
{
    const COLLECTION_METHOD_POINT_OF_SALE = 'pos';
    const COLLECTION_METHOD_AFTER_FULLFILLMENT = 'af';

    /**
     * @var Logger
     */
    private $logger;

    /**
     * @var string
     */
    private $pluginDir;

    /**
     * Frontend constructor.
     *
     * @param Logger $logger
     * @param $pluginDir
     */
    public function __construct(
        Logger $logger,
        $pluginDir
    ) {
        $this->logger = $logger;
        $this->pluginDir = $pluginDir;
    }

    public static function getSubscribedEvents()
    {
        return [
            'Enlight_Controller_Action_PostDispatchSecure_Frontend' => 'onPostDispatch',
            'Shopware_CronJob_leopardAusgezeichnetOrgCron' => 'onCron',
            'Shopware_Modules_Order_SaveOrder_ProcessDetails' => 'onOrderSave',
            'Shopware_Controllers_Frontend_Checkout::finishAction::after' => 'onCheckoutFinish',
            'Theme_Inheritance_Template_Directories_Collected' => 'onCollectTemplateDir',
        ];
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onOrderSave(Enlight_Event_EventArgs $args)
    {
        $config = Shopware()->Container()->get('shopware.plugin.config_reader')->getByPluginName('leopardAusgezeichnetOrg', Shopware()->Shop());
        $order = Shopware()->Models()->getRepository('Shopware\Models\Order\Order')->find($args->get('orderId'));

        if (!$config['leopardAusgezeichnetOrgActivate'] ||
            $config['leopardAusgezeichnetOrgMailTime'] != 2 ||
            !$order instanceof Order ||
            !$order->getCustomer() instanceof Customer) {
            return;
        }

        try {
            // send mail
            $this->sendMail($order, Shopware()->Shop(), $config['leopardAusgezeichnetOrgRatingMailTemplate']);

            // set status
            $order->getAttribute()->setleopardAusgezeichnetOrgStatus(true);
            Shopware()->Models()->persist($order);
            Shopware()->Models()->flush();
        } catch (Exception $e) {
            $this->logger->addError($e->getMessage());
        }
    }

    /**
     * @param Enlight_Hook_HookArgs $args
     */
    public function onCheckoutFinish(Enlight_Hook_HookArgs $args)
    {
        $config = Shopware()->Container()->get('shopware.plugin.config_reader')->getByPluginName('leopardAusgezeichnetOrg', Shopware()->Shop());

        if (!$config['leopardAusgezeichnetOrgActivate'] ||
            !$config['leopardAusgezeichnetOrgReviewCollector'] ||
            !$config['leopardAusgezeichnetOrgCodeReview']) {
            return;
        }

        /** @var Enlight_View_Default $view */
        $view = $args->getSubject()->View();
        $view->assign('leopard_ausgezeichnet_org_review_collector', $config['leopardAusgezeichnetOrgCodeReview']);
    }

    public function onCron()
    {
        $shops = Shopware()->Models()->getRepository('Shopware\Models\Shop\Shop')->findAll();

        if (empty($shops)) {
            return;
        }

        $count = 0;
        $shop_config = [];
        foreach ($shops as $shop) {
            $shop_config[$count]['shop'] = $shop;
            $shop_config[$count]['config'] = Shopware()->Container()->get('shopware.plugin.config_reader')->getByPluginName('leopardAusgezeichnetOrg', $shop);
            ++$count;
        }

        if (empty($shop_config)) {
            return;
        }

        foreach ($shop_config as $config) {
            if (!$config['config']['leopardAusgezeichnetOrgActivate'] ||
                !$config['config']['leopardAusgezeichnetOrgMailTimeDay']) {
                continue;
            }

            /** @var QueryBuilder $baseQueryBuilder */
            $baseQueryBuilder = Shopware()->Container()->get('dbal_connection')->createQueryBuilder();
            $ordersQuery = '';

            // "X" days after shipment
            if ($config['config']['leopardAusgezeichnetOrgMailTime'] == 1) {
                // get orders
                $ordersQuery = $baseQueryBuilder
                    ->select('o.id')
                    ->from('s_order', 'o')
                    ->innerJoin('o', 's_order_attributes', 'oa', 'o.id = oa.orderID')
                    ->orwhere('o.status = 2')
                    ->orWhere('o.status = 7')
                    ->andWhere('o.language = ' . $config['shop']->getId())
                    ->andWhere('datediff(now(), o.ordertime) = ' . $config['config']['leopardAusgezeichnetOrgMailTimeDay'])
                    ->andWhere('oa.leopard_ausgezeichnet_org_status is null or oa.leopard_ausgezeichnet_org_status = 0')
                    ->orderBy('o.ordernumber', 'ASC');
            }

            // "X" days after status change (completely finished / completely delivered)
            if ($config['config']['leopardAusgezeichnetOrgMailTime'] == 3) {
                // get orders
                $ordersQuery = $baseQueryBuilder
                    ->select('o.id')
                    ->from('s_order', 'o')
                    ->innerJoin('o', 's_order_attributes', 'oa', 'o.id = oa.orderID')
                    ->innerJoin('o', 's_order_history', 'oh', 'o.id = oh.orderID')
                    ->orwhere('o.status = 2')
                    ->orWhere('o.status = 7')
                    ->andWhere('o.language = ' . $config['shop']->getId())
                    ->andWhere('datediff(now(), (SELECT MAX(change_date) FROM s_order_history WHERE orderID = o.id)) = ' . $config['config']['leopardAusgezeichnetOrgMailTimeDay'])
                    ->andWhere('oa.leopard_ausgezeichnet_org_status is null or oa.leopard_ausgezeichnet_org_status = 0')
                    ->groupBy('o.ordernumber')
                    ->orderBy('o.ordernumber', 'ASC');
            }

            if (!$ordersQuery instanceof QueryBuilder) {
                continue;
            }

            $orders = $ordersQuery->execute()->fetchAll(PDO::FETCH_ASSOC);

            if (empty($orders)) {
                continue;
            }

            foreach ($orders as $value) {
                $order = Shopware()->Models()->getRepository('Shopware\Models\Order\Order')->find($value['id']);
                if (!$order instanceof Order ||
                    !$order->getCustomer() instanceof Customer) {
                    continue;
                }

                try {
                    // send mail
                    $this->sendMail($order, $config['shop'], $config['config']['leopardAusgezeichnetOrgRatingMailTemplate'], self::COLLECTION_METHOD_AFTER_FULLFILLMENT);

                    // set status
                    $order->getAttribute()->setleopardAusgezeichnetOrgStatus(true);

                    // writing changes
                    Shopware()->Models()->persist($order);
                    Shopware()->Models()->flush();
                } catch (Exception $e) {
                    $this->logger->addError($e->getMessage());
                }
            }
        }
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onPostDispatch(Enlight_Event_EventArgs $args)
    {
        $config = Shopware()->Container()->get('shopware.plugin.config_reader')->getByPluginName('leopardAusgezeichnetOrg', Shopware()->Shop());

        if (!$config['leopardAusgezeichnetOrgActivate'] ||
            !$config['leopardAusgezeichnetOrgSeal'] ||
            !$config['leopardAusgezeichnetOrgSealCode']) {
            return;
        }

        /** @var Enlight_View_Default $view */
        $view = $args->getSubject()->View();
        $view->assign('ausgezeichnet_org_seal_code', $config['leopardAusgezeichnetOrgSealCode']);
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onCollectTemplateDir(Enlight_Event_EventArgs $args)
    {
        $dirs = $args->getReturn();
        $dirs[] = $this->pluginDir . '/Resources/views/';

        $args->setReturn($dirs);
    }

    /**
     * @param Order $order
     * @param $method
     * @param $shop
     *
     * @return string|void
     */
    private function buildRatingLink(
        Order $order,
        $method,
        $shop
    ) {
        if (!$order instanceof Order ||
            !is_string($method) ||
            !$shop instanceof Shop) {
            return;
        }

        $config = Shopware()->Container()->get('shopware.plugin.config_reader')->getByPluginName('leopardAusgezeichnetOrg', $shop);
        $hash = md5($config['leopardAusgezeichnetOrgCustomerNumber'] . '/' . $order->getNumber() . '/' . $method . '/' . $config['leopardAusgezeichnetOrgIndividualRatingID']);

        return $config['leopardAusgezeichnetOrgProfilLink'] . '?txid=' . $order->getNumber() . '&collmethod=' . $method . '&hash=' . $hash;
    }

    /**
     * @param Order $order
     * @param $shop
     * @param string $mailTemplate
     * @param string $method
     */
    private function sendMail(
        Order $order,
        $shop,
        $mailTemplate = 'ausgezeichnetorgEmail',
        $method = self::COLLECTION_METHOD_POINT_OF_SALE
    ) {
        if (!$order instanceof Order ||
            !$shop instanceof Shop) {
            return;
        }

        $mailTemplate = Shopware()->Models()->getRepository('Shopware\Models\Mail\Mail')->findOneBy(['name' => $mailTemplate]);

        if (!$mailTemplate instanceof Mail) {
            $mailTemplate = Shopware()->Models()->getRepository('Shopware\Models\Mail\Mail')->findOneBy(['name' => 'ausgezeichnetorgEmail']);
        }

        // set mail vars
        $vars = [
            'leopard_ausgezeichnet_org' => [
                'ordernumber' => $order->getNumber(),
                'orderdate' => date('d.m.Y', $order->getOrderTime()->getTimestamp()),
                'ratinglink' => $this->buildRatingLink($order, $method, $shop),
            ],
        ];

        // build mail
        try {
            $mail = Shopware()->TemplateMail()->createMail($mailTemplate->getName(), $vars, $shop);
            $subject = $mail->getSubject();
            $mail->clearSubject();
            $mail->setSubject($subject);
            $mail->addTo($order->getCustomer()->getEmail());

            // send mail
            $mail->send();
        } catch (Exception $e) {
            $this->logger->addError($e->getMessage());
        }
    }
}
